function  [SPF, real, hist] = load_data(variable, sample)

% Loads realised data and SPF forecast data

% INPUT
% variable:     variable name
% sample:       dates for sample considered

% OUTPUT
% SPF:          SPF density forecasts (structure)
% real:         Realised data (structure)
% hist:         Historical realisation of variable for benchmark forecasts construction (structure)

%% load SPF data
spf = importdata(['SPF',variable,'.xlsx']);

dates_surveys = datetime(spf.textdata.x12M(2:end,2),'InputFormat','yyyyQQQ');
dates_start = find(dates_surveys>=sample(1),1,"first");
dates_end = find(dates_surveys<=sample(2),1,"last");

SPF.deadlines = datetime(spf.textdata.x12M(2:end,3),'InputFormat','dd/MM/yyyy');

% data contains all surveys (T x B x H)
SPF.data(:,:,1) = spf.data.x12M(dates_start:dates_end, 4:end)./100;
SPF.data(:,:,2) = spf.data.x24M(dates_start:dates_end, 4:end)./100;

SPF.edges = spf.data.edges(dates_start:dates_end, 3:end);
SPF.deadlines = SPF.deadlines(dates_start:dates_end);
SPF.dates_surveys = dates_surveys(dates_start:dates_end);

% load also lagged surveys (with bin adjustment)
SPF.naive(:,:,1) = spf.data.B12M(dates_start-1:dates_end-1, :)./100;
SPF.naive(:,:,2) = spf.data.B24M(dates_start-1:dates_end-1, :)./100;

%% load realised
[rls.data,  rls.textdata]= xlsread('realised.xlsx', variable);
rls.data = flip(rls.data,1); % flip the dates
rls.data = flip(rls.data,2); % flip the vintages

%% select sample
try
    dates_realised = flip(datetime(rls.textdata(2:end,1),'InputFormat','yyyyMMM'));
catch
    dates_realised = flip(datetime(rls.textdata(2:end,1),'InputFormat','yyyyQQQ'));
end

dates_start_r = find(dates_realised<=SPF.dates_surveys(1),1,"last");
dates_end_r = find(dates_realised>=SPF.dates_surveys(end),1,"first");

%% historical values to compute benchmarks
hist.vintages = flip(datetime(rls.textdata(1, 2:end), 'InputFormat','dd/MM/yyyy HH:mm:ss'));
hist.data = rls.data;
hist.dates_realised = dates_realised;

%% realised values for forecast evaluation

% shift realised for forecast horizon
switch variable
    case 'UN'
        shift = 10;
        f = 3; % 3 obs per quarter
    case 'HICP'
        shift = 11;
         f = 3; % 3 obs per quarter
    case 'GDP'
        shift = 2;
        f = 1; % 1 obs per quarter
end

for h=[12 24]
    real.realised_all(:,:,h/12) = rls.data(dates_start_r+(h-12)*(f==3)+ ...
        (h/3-4)*(f==1)+shift : f : dates_end_r+(h-12)*(f==3)+ (h/3-4)*(f==1) +shift,:);

    for t=1:size(real.realised_all,1)
        temp = real.realised_all(t,~isnan(real.realised_all(t,:,h/12)),h/12);
        real.realised(t,1,h/12) = temp(1);
    end
end
end




